<?php
session_start();

/*
 * Single Sign-On (SSO) style admin for go.ownpaste.com
 * This expects MgBee to create a JSON token file with:
 *  { "token": "....", "expires": 1234567890 }
 *
 * Then MgBee links to:
 *  https://go.ownpaste.com/admin.php?auth=<token>
 */

// EDIT THIS: shared token file path that BOTH mgbee.xyz and go.ownpaste.com can read.
const LS_SSO_TOKEN_FILE = '/home/mobileth/shared-admin-token.json'; // <- change to your real path

// ---------- Helper: load links ----------
function ls_load_links(): array {
    $file = __DIR__ . '/links.json';
    if (!file_exists($file)) {
        return [];
    }
    $json = file_get_contents($file);
    $data = json_decode($json, true);
    if (!is_array($data)) {
        return [];
    }
    return $data;
}

// ---------- Helper: save links ----------
function ls_save_links(array $links): bool {
    $file = __DIR__ . '/links.json';
    $json = json_encode($links, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    return (bool) file_put_contents($file, $json, LOCK_EX);
}

// ---------- Helper: generate random ID ----------
function ls_generate_id(int $length = 6): string {
    $chars = 'abcdefghijklmnopqrstuvwxyz0123456789';
    $id = '';
    for ($i = 0; $i < $length; $i++) {
        $id .= $chars[random_int(0, strlen($chars) - 1)];
    }
    return $id;
}

// ---------- Helper: normalize a link entry (for stats) ----------
// Supports both formats:
//   "id": "https://url..."                        (old)
//   "id": { "url": "...", "clicks": N, "last_used": "..." }  (new)
function ls_normalize_entry($entry): array {
    if (is_string($entry)) {
        return [
            'url'       => $entry,
            'clicks'    => 0,
            'last_used' => null,
        ];
    }
    if (!is_array($entry)) {
        return [
            'url'       => '',
            'clicks'    => 0,
            'last_used' => null,
        ];
    }

    $url       = isset($entry['url']) ? (string)$entry['url'] : '';
    $clicks    = isset($entry['clicks']) && is_numeric($entry['clicks']) ? (int)$entry['clicks'] : 0;
    $lastUsed  = isset($entry['last_used']) ? (string)$entry['last_used'] : null;

    return [
        'url'       => $url,
        'clicks'    => $clicks,
        'last_used' => $lastUsed,
    ];
}

/*
 *  SSO LOGIN HANDLING
 *  If session is not yet admin, require a valid ?auth= token from MgBee.
 */

if (!isset($_SESSION['ls_admin'])) {

    // If we have auth token in URL, validate it
    if (isset($_GET['auth'])) {
        $auth = $_GET['auth'];

        if (!file_exists(LS_SSO_TOKEN_FILE)) {
            die('SSO token file not found. Open this panel from MgBee admin.');
        }

        $json = file_get_contents(LS_SSO_TOKEN_FILE);
        $data = json_decode($json, true);

        if (!is_array($data) || !isset($data['token'], $data['expires'])) {
            die('Invalid SSO token file.');
        }

        if (!hash_equals($data['token'], $auth)) {
            die('Invalid SSO token.');
        }

        if (time() > (int)$data['expires']) {
            die('SSO token expired. Please open from MgBee admin again.');
        }

        // Token is valid -> log in admin
        $_SESSION['ls_admin'] = true;

        // Optional: one time use only
        @unlink(LS_SSO_TOKEN_FILE);

        header('Location: admin.php');
        exit;
    }

    // No session and no auth token: deny direct access
    die('Unauthorized. You must open this admin from MgBee admin panel.');
}

// ---------- LOGOUT ----------
if (isset($_GET['logout'])) {
    unset($_SESSION['ls_admin']);
    header('Location: admin.php');
    exit;
}

// ---------- AUTHENTICATED SECTION ----------

$links   = ls_load_links();
$message = '';

// Handle ADD new link
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add') {
    $custom_id = trim($_POST['short_id'] ?? '');
    $target    = trim($_POST['target'] ?? '');

    if (empty($target) || !filter_var($target, FILTER_VALIDATE_URL)) {
        $message = 'Please enter a valid target URL.';
    } else {
        // Generate or sanitize ID
        if ($custom_id === '') {
            // auto id
            do {
                $new_id = ls_generate_id(6);
            } while (isset($links[$new_id]));
        } else {
            // use custom id, only allow simple chars
            $new_id = preg_replace('/[^a-zA-Z0-9_-]/', '', $custom_id);
            if ($new_id === '') {
                $message = 'Invalid custom ID.';
            } elseif (isset($links[$new_id])) {
                $message = 'This ID already exists.';
            }
        }

        if ($message === '') {
            // Store as simple string; go.php will upgrade to array with stats on first use
            $links[$new_id] = $target;

            if (ls_save_links($links)) {
                $message = 'Link added: ' . $new_id;
            } else {
                $message = 'Failed to save links.json';
            }
        }
    }
}

// Handle DELETE link
if (isset($_GET['delete'])) {
    $del_id = $_GET['delete'];
    if (isset($links[$del_id])) {
        unset($links[$del_id]);
        if (ls_save_links($links)) {
            $message = 'Deleted link: ' . htmlspecialchars($del_id);
        } else {
            $message = 'Failed to save after delete.';
        }
        // Reload links after save
        $links = ls_load_links();
    } else {
        $message = 'ID not found.';
    }
}

// Basic stats
$total_links = count($links);

// Build base short URL (uses /s/ID)
$host   = $_SERVER['HTTP_HOST'] ?? 'go.ownpaste.com';
$scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
$base_short = $scheme . '://' . $host . '/s/';

?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <title>Go.ownpaste Shortener Admin</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
</head>
<body style="background:#020617; color:#e5e7eb; font-family:Arial,system-ui;">

<div style="max-width:960px; margin:20px auto; padding:20px;">
    <div style="display:flex; justify-content:space-between; align-items:center;">
        <div>
            <h2 style="margin:0;">Go.ownpaste Shortener Admin</h2>
            <div style="margin-top:4px; font-size:12px; color:#9ca3af;">
                Total links: <strong><?php echo (int)$total_links; ?></strong> &middot;
                Base short URL: <strong><?php echo htmlspecialchars($base_short); ?></strong>
            </div>
        </div>
        <a href="admin.php?logout=1" style="color:#f97316;">Logout</a>
    </div>

    <?php if ($message): ?>
        <p style="margin-top:15px; padding:10px; background:#111827; border-radius:6px;">
            <?=htmlspecialchars($message)?>
        </p>
    <?php endif; ?>

    <!-- Add new link form -->
    <div style="margin-top:20px; padding:15px; background:#0b1120; border-radius:10px;">
        <h3 style="margin-top:0;">Create New Short Link</h3>
        <form method="post">
            <input type="hidden" name="action" value="add">

            <div style="margin-bottom:10px;">
                <label>Custom ID (optional):</label><br>
                <input type="text" name="short_id" placeholder="e.g. abc123 (leave empty for auto)"
                       style="width:100%; padding:8px; border-radius:6px; border:1px solid #374151; background:#020617; color:#e5e7eb;">
            </div>

            <div style="margin-bottom:10px;">
                <label>Target URL:</label><br>
                <input type="text" name="target" placeholder="https://mega.nz/file/xxxx"
                       style="width:100%; padding:8px; border-radius:6px; border:1px solid #374151; background:#020617; color:#e5e7eb;">
            </div>

            <button type="submit"
                    style="padding:8px 20px; background:#22c55e; border:none; border-radius:6px; color:#fff; font-weight:bold; cursor:pointer;">
                Save Link
            </button>
        </form>
    </div>

    <!-- List existing links -->
    <div style="margin-top:20px; padding:15px; background:#020617; border-radius:10px; border:1px solid #1f2937;">
        <h3 style="margin-top:0;">Existing Links (<?=count($links)?>)</h3>
        <?php if (empty($links)): ?>
            <p>No links yet.</p>
        <?php else: ?>
            <table style="width:100%; border-collapse:collapse; font-size:14px;">
                <thead>
                    <tr style="background:#111827;">
                        <th style="padding:8px; border-bottom:1px solid #374151; text-align:left;">ID</th>
                        <th style="padding:8px; border-bottom:1px solid #374151; text-align:left;">Short URL</th>
                        <th style="padding:8px; border-bottom:1px solid #374151; text-align:left;">Target URL</th>
                        <th style="padding:8px; border-bottom:1px solid #374151; text-align:right;">Clicks</th>
                        <th style="padding:8px; border-bottom:1px solid #374151; text-align:left;">Last used</th>
                        <th style="padding:8px; border-bottom:1px solid #374151;">Action</th>
                    </tr>
                </thead>
                <tbody>
                <?php foreach ($links as $id => $rawEntry): ?>
                    <?php
                        $entry    = ls_normalize_entry($rawEntry);
                        $url      = $entry['url'];
                        $clicks   = (int)$entry['clicks'];
                        $lastUsed = $entry['last_used'] ?: '—';
                        $shortUrl = $base_short . $id;
                    ?>
                    <tr>
                        <td style="padding:6px; border-bottom:1px solid #1f2937;"><?=htmlspecialchars($id)?></td>
                        <td style="padding:6px; border-bottom:1px solid #1f2937;">
                            <a href="/s/<?=urlencode($id)?>" target="_blank"
                               style="color:#60a5fa;">
                                <?=htmlspecialchars($shortUrl)?>
                            </a>
                        </td>
                        <td style="padding:6px; border-bottom:1px solid #1f2937; max-width:350px; word-wrap:break-word;">
                            <?=htmlspecialchars($url)?>
                        </td>
                        <td style="padding:6px; border-bottom:1px solid #1f2937; text-align:right;">
                            <?= $clicks ?>
                        </td>
                        <td style="padding:6px; border-bottom:1px solid #1f2937;">
                            <?=htmlspecialchars($lastUsed)?>
                        </td>
                        <td style="padding:6px; border-bottom:1px solid #1f2937; text-align:center;">
                            <a href="admin.php?delete=<?=urlencode($id)?>"
                               onclick="return confirm('Delete this link?');"
                               style="color:#f97316;">
                               Delete
                            </a>
                        </td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
</div>

</body>
</html>
