<?php
session_start();

// Require id + key (token from step2)
if (!isset($_GET['id'], $_GET['key'])) {
    http_response_code(400);
    die("Invalid request.");
}

$id  = $_GET['id'];
$key = $_GET['key'];

// Validate Step 2 token (set in step2.php)
if (!isset($_SESSION["step2_token_$id"]) || $key !== $_SESSION["step2_token_$id"]) {
    http_response_code(403);
    die("Unauthorized access.");
}

$linksFile = __DIR__ . '/links.json';
if (!file_exists($linksFile)) {
    http_response_code(500);
    die("Links store missing.");
}

$linksRaw = file_get_contents($linksFile);
$links    = json_decode($linksRaw, true);

if (!is_array($links) || !isset($links[$id])) {
    http_response_code(404);
    die("Final link not found.");
}

// --------------------------------------
// Backward compatible structure:
// Old: "id": "https://target-url.com"
// New: "id": { "url": "...", "clicks": 0, "last_used": "..." }
// --------------------------------------
$entry = $links[$id];

if (is_string($entry)) {
    // Old simple format – convert on the fly
    $entry = [
        'url'       => $entry,
        'clicks'    => 0,
        'last_used' => null,
    ];
}

$finalUrl = isset($entry['url']) ? $entry['url'] : null;
if (!$finalUrl) {
    http_response_code(500);
    die("Invalid link record.");
}

// Update stats
$entry['clicks']    = isset($entry['clicks']) && is_numeric($entry['clicks'])
    ? (int)$entry['clicks'] + 1
    : 1;
$entry['last_used'] = date('Y-m-d H:i:s');

// Save back to array
$links[$id] = $entry;

// Try saving stats back into links.json
file_put_contents(
    $linksFile,
    json_encode($links, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES),
    LOCK_EX
);

// Optional: make token one-time use only
unset($_SESSION["step2_token_$id"]);
?>
<!DOCTYPE html>
<html>
<head>
    <title>Redirecting… - MgBee Gateway</title>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">

    <!-- Fallback meta refresh (3 seconds) -->
    <meta http-equiv="refresh" content="3;url=<?php echo htmlspecialchars($finalUrl, ENT_QUOTES); ?>">

    <style>
        * { box-sizing: border-box; }

        body {
            margin: 0;
            min-height: 100vh;
            font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
            background: radial-gradient(circle at top, #020617 0, #000 70%);
            color: #e5e7eb;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }

        .card-wrap {
            position: relative;
            width: 100%;
            max-width: 380px;
        }

        .glow-ring {
            position: absolute;
            inset: -2px;
            border-radius: 22px;
            background: conic-gradient(from 180deg, #22c55e, #0ea5e9, #22c55e);
            opacity: 0.7;
            filter: blur(10px);
            z-index: 0;
            animation: spinGlow 6s linear infinite;
        }

        @keyframes spinGlow {
            to { transform: rotate(360deg); }
        }

        .card {
            position: relative;
            z-index: 1;
            background: rgba(15, 23, 42, 0.97);
            border-radius: 18px;
            padding: 26px 22px 20px;
            border: 1px solid rgba(148,163,184,0.18);
            box-shadow:
                0 22px 40px rgba(0,0,0,0.85),
                0 0 0 1px rgba(15,23,42,0.9);
            text-align: center;
        }

        .badge {
            display: inline-flex;
            align-items: center;
            gap: 6px;
            font-size: 11px;
            text-transform: uppercase;
            letter-spacing: .18em;
            padding: 4px 10px;
            border-radius: 999px;
            border: 1px solid rgba(148,163,184,0.4);
            color: #a5b4fc;
            background: radial-gradient(circle at top left, rgba(59,130,246,0.35), transparent 60%);
        }

        .dot {
            width: 6px;
            height: 6px;
            border-radius: 999px;
            background: #22c55e;
            box-shadow: 0 0 8px #22c55e;
        }

        h2 {
            margin: 14px 0 6px;
            font-size: 21px;
            color: #f9fafb;
        }

        .sub {
            margin: 0 0 18px;
            font-size: 14px;
            color: #94a3b8;
        }

        .spinner-wrap {
            display:flex;
            flex-direction:column;
            align-items:center;
            justify-content:center;
            gap:10px;
            margin-bottom:14px;
        }

        .spinner {
            width: 36px;
            height: 36px;
            border-radius: 999px;
            border: 3px solid rgba(148,163,184,0.3);
            border-top-color: #22c55e;
            border-right-color: #3b82f6;
            animation: spin 0.8s linear infinite;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        .redirect-info {
            font-size: 12px;
            color: #9ca3af;
        }

        .redirect-info strong {
            color: #e5e7eb;
        }

        .manual-link {
            margin-top: 12px;
            font-size: 12px;
            color: #64748b;
        }

        .manual-link a {
            color: #93c5fd;
            text-decoration: none;
            word-break: break-all;
        }

        .manual-link a:hover {
            text-decoration: underline;
        }

        .foot {
            margin-top: 10px;
            font-size: 11px;
            color: #6b7280;
        }

        .tiny-dot {
            display:inline-block;
            width:7px;
            height:7px;
            border-radius:999px;
            background:#22c55e;
            box-shadow:0 0 8px rgba(34,197,94,0.9);
            margin-right:4px;
        }

        @media (max-width: 480px) {
            .card { padding: 22px 18px 18px; }
        }
    </style>

    <script>
        const targetUrl = <?php echo json_encode($finalUrl); ?>;

        function startRedirect() {
            // small delay for UX (spinner visible)
            setTimeout(function () {
                window.location.href = targetUrl;
            }, 2000);
        }
    </script>
</head>
<body onload="startRedirect()">

<div class="card-wrap">
    <div class="glow-ring"></div>
    <div class="card">
        <div class="badge">
            <span class="dot"></span>
            Redirecting
        </div>

        <h2>Opening Your Link…</h2>
        <p class="sub">
            Your unlock was verified successfully. We are now redirecting you to the final destination.
        </p>

        <div class="spinner-wrap">
            <div class="spinner"></div>
            <div class="redirect-info">
                <strong>Please wait a moment…</strong><br>
                If nothing happens, use the manual link below.
            </div>
        </div>

        <div class="manual-link">
            Or click here: <br>
            <a href="<?php echo htmlspecialchars($finalUrl, ENT_QUOTES); ?>">
                <?php echo htmlspecialchars($finalUrl, ENT_QUOTES); ?>
            </a>
        </div>

        <div class="foot">
            <span class="tiny-dot"></span> MgBee Gateway - Secure one-time redirect
        </div>
    </div>
</div>

</body>
</html>
